using System.Text.RegularExpressions;

namespace gov.va.med.vbecs.Common
{

	#region Header

	///<Package>Package: VBECS - VistA Blood Establishment Computer System</Package>
	///<Warning> WARNING: Per VHA Directive $VADIRECTIVE this class should not be modified</Warning>
	///<MedicalDevice> Medical Device #: $MEDDEVICENO</MedicalDevice>
	///<Developers>
	///	<Developer>Krzysztof Dobranowski</Developer>
	///</Developers>
	///<SiteName>Hines OIFO</SiteName>
	///<CreationDate>2/24/2003</CreationDate>
	///<Note>The Food and Drug Administration classifies this software as a medical device.  As such, it may not be changed in any way. Modifications to this software may result in an adulterated medical device under 21CFR820, the use of which is considered to be a violation of US Federal Statutes.  Acquiring and implementing this software through the Freedom of information Act requires the implementor to assume total responsibility for the software, and become a registered manufacturer of a medical device, subject to FDA regulations</Note>
	///<summary>RegularExpressions class</summary>

	#endregion

	public class RegularExpressions
	{
		/*
		 *	*        - 0 or more matches, same as {0,}
		 *  +        - 1 or more matches, same as {1,}
		 *  ?        - 0 or 1 matches, same as {0,1}
		 *  {n}      - exactly n matches
		 *  {n,}     - n or more matches
		 *  {n,m}    - between n and m matches
		 *  ^        - the beginning of the string, or line
		 *  $        - the end of the string, or line
		 *  .        - any character except new line character
		 *  [aeiou]  - any character between square brackets; [0-9] matches any digit
		 *  [^aeiou] - any character except those between square bracket
		 *  \w       - a word character, same as [A-Za-z0-9]
		 *  \W       - a non-word character
		 *  \d       - a digit, same as [0-9]
		 *  \D       - a non digit
		 *  \s       - a white space character (space, tab, form-feed, newline, or carraige return)
		 *  \S       - a non white space character
		 *  \b       - word boundary
		 *  \B       - not a word boundary
		 */

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/24/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2298"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Regex object</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="531"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns the regular expression used to determine if a unit id is Codabar.
		/// </summary>
		/// <returns></returns>
		public static Regex CodabarUnitID()
		{
			//Changed because pooled units can be 11 digits
			//return new Regex("^[A-Z0-9]{2}[0-9]{5,7}$");
			return new Regex(@"^([0-9]{2})?[A-Z0-9]{1,2}[0-9]{5,8}[A-Z]{0,1}$");
		}
		
		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/24/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="533"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Regex object</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="534"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// ScannedCodabarUnitID
		/// </summary>
		/// <returns></returns>
		public static Regex ScannedCodabarUnitID()
		{
			return new Regex(@"^[0-9]{6,10}$");
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/24/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2299"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Regex object</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="535"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns the regular expression used to determine if a unit id is ISBT128.
		/// </summary>
		/// <returns></returns>
		public static Regex ScannedISBT128UnitID()
		{
			//Changed because pooled units can be 1 alpha, 10 digits
			return new Regex("^=?[A-Z]{1}[0-9]{12,14}$");
		}
		
		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/24/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="541"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Regex object</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="544"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// ISBT128UnitID
		/// </summary>
		/// <returns></returns>
		public static Regex ISBT128UnitID()
		{
			//Changed because pooled units can be 1 alpha, 10 digits
			return new Regex("[A-Z]{1}[0-9]{12}$");
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/24/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2300"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Regex object</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="546"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns the regular expression used to determine if a string is a correctly formatted
		/// FDA registration number.
		/// </summary>
		/// <returns></returns>
		public static Regex FDARegistrationNumber()
		{
			return new Regex(@"(^\d{7}$)|(^\d{10}$)");
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/24/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2301"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Regex object</ExpectedOutput>
		///	</Case>
		///<Case type="1" testid ="547"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns the regular expression used to determine if a product code is of type Codabar.
		/// </summary>
		/// <returns></returns>
		public static Regex CodabarProductCode()
		{
			return new Regex("^[0-9]{5,7}$");
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/24/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2302"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Regex object</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="551"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns the regular expression used to determine if a product code is of type ISBT128.
		/// </summary>
		/// <returns></returns>
		public static Regex ISBT128ProductCode()
		{
			//CR 2756 Per BR_25.07 changed to allow the last character to be a lower case alpha
			return new Regex("^=?<?[A-Z]{1}[0-9]{4}[(?i:A-Za-z)0-9]{0,1}[A-Z0-9]{0,1}[a-z0-9]{0,1}$");
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/24/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2303"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Regex object</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="552"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns the regular expression used to determine if the division ID is valid.
		/// </summary>
		/// <returns></returns>
		public static Regex UnitDivisionId()
		{
			return new Regex("^[A-Z]{1}[0-9]{1}$");
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/24/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2304"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Regex object</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="553"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns the regular expression used to determine if the numeric threshold is valid.
		/// </summary>
		/// <returns></returns>
		public static Regex Threshold()
		{
			return new Regex(@"^[<>]([ ]{0,1})([0-9]{1,3})(([.]{1}[0-9]{0,2}){0,1})$");
		}

		///<Developers>
		///	<Developer>Cameron Taylor</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/6/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3119"> 
		///		<ExpectedInput>Valid threshold string (1-15 chars)</ExpectedInput>
		///		<ExpectedOutput>True regular expression match indicator</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="3120"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns the regular expression used to determine if the text threshold is valid.
		/// </summary>
		/// <returns></returns>
		public static Regex ThresholdText()
		{
			return new Regex(@"^[A-Za-z0-9=+, ]{1,255}$");
		}

		///<Developers>
		///	<Developer>Krzysztof Dobranowski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>2/24/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2305"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Regex</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="554"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns the regular expression used to determine if the ICCBA registration number is valid.
		/// </summary>
		/// <returns></returns>
		public static Regex IccbbaRegistrationNumber()
		{
			return new Regex("^[A-Z]{1}[0-9]{4}$");
		}

		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/26/2003</CreationDate>
		///<TestCases>
		///	<Case type="0" testid ="3069"> 
		///		<ExpectedInput>Valid phone numbers</ExpectedInput>
		///		<ExpectedOutput>Valid Regex</ExpectedOutput>
		///	</Case>		
		///	<Case type="1" testid ="3070"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns the regular expression used to determine if a phone number (with area code)
		/// It works with delimiters or no delimiters (i.e. 111-222-3333, or 111.222.3333, or (111) 222-3333, 
		/// or 1112223333, etc...).  
		/// </summary>
		/// <returns>Regex</returns>
		public static Regex PhoneNumber()
		{
			return new Regex(@"^\D?(\d{3})\D?\D?(\d{3})\D?(\d{4})$");
		}
		
		///<Developers>
		///	<Developer>Luke Meyer</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/26/2003</CreationDate>
		///<TestCases>
		///	<Case type="0" testid ="3067"> 
		///		<ExpectedInput>Valid zip code</ExpectedInput>
		///		<ExpectedOutput>Valid Regex</ExpectedOutput>
		///	</Case>		
		///	<Case type="1" testid ="3068"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns the regular expression used to determine if a zip code is valid.
		/// It matches standard 5 digit or ZIP + 4 zip codes. 
		/// </summary>
		/// <returns>Regex</returns>
		public static Regex ZipCode()
		{
			return new Regex(@"^\d{5}(-\d{4})?$");
		}
		
		///<Developers>
		///	<Developer>John Vrooland</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/13/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="2894"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Regex for Email addresses</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="2895"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns the regular expression used to determine if an email address is valid.
		/// </summary>
		/// <returns>Regex</returns>
		public static Regex EmailAddress()
		{			
			return new Regex( @"^((?>[a-zA-Z\d!#$%&'*+\-/=?^_`{|}~]+\x20*|""((?=[\x01-\x7f])[^""\\]|\\[\x01-\x7f])*""\x20*)*(?<angle><))?((?!\.)(?>\.?[a-zA-Z\d!#$%&'*+\-/=?^_`{|}~]+)+|""((?=[\x01-\x7f])[^""\\]|\\[\x01-\x7f])*"")@(((?!-)[a-zA-Z\d\-]+(?<!-)\.)+[a-zA-Z]{2,}|\[(((?(?<!\[)\.)(25[0-5]|2[0-4]\d|[01]?\d?\d)){4}|[a-zA-Z\d\-]*[a-zA-Z\d]:((?=[\x01-\x7f])[^\\\[\]]|\\[\x01-\x7f])+)\])(?(angle)>)$" );
		}
		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="555"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>RegEx object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="556"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// EyeReadablePrefixNumeric
		/// </summary>
		/// <returns></returns>
		public static Regex EyeReadablePrefixNumeric()
		{
			return new Regex(@"^[0-9]{2}");
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>11/2/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="5919"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>RegEx object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="5920"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Determines if the blood unit has been split already by checking if last character is alpha
		/// </summary>
		/// <returns></returns>
		public static Regex EyeReadableSuffixAlpha()
		{
			return new Regex(@"^[A-Z]{1}");
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>6/13/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="1759"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>RegEx object</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="1760"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// EyeReadablePrefixAlphaNumeric
		/// </summary>
		/// <returns></returns>
		public static Regex EyeReadablePrefixAlphaNumeric()
		{
			return new Regex(@"[A-Z0-9]{2}");
		}

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/29/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8130"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Regex object</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8131"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// PooledCodabarUnitId
		/// </summary>
		/// <returns></returns>
		public static Regex PooledCodabarEyeReadableUnitId()
		{
			return new Regex(@"^[A-Za-z0-9]{6,13}");
		}


		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/4/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3436"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Regex</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="557"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns the regular expression used to determine if a ABO/Rh barcode label is Codabar.
		/// </summary>
		/// <returns>Regex</returns>
		public static Regex CodabarABORh()
		{
			return new Regex("[A-Z0-9]{1}[0-9]{1,2}");
		}
		
		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/4/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3440"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Regex</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="558"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns the regular expression used to determine if a Expiration Date is Codabar.
		/// Allowable formats: mmddyy or mmddyyyy where mm = month, dd = day, yy or yyyy = year 
		/// </summary>
		/// <returns></returns>
		public static Regex CodabarExpirationDate()
		{
			return new Regex("[0-9]{6,8}");
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/4/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3442"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Regex</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="559"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns the regular expression used to determine if a Expiration Date is ISBT128.
		/// Allowable formats: cyyjjj, cyyjjjhhmm where
		///		c = century indicator (9 for 1999, 0 for 2000 etc.)
		///		yy = year (i.e. 02 for 2002)
		///		jjj = julian date (the number of the day in the year, i.e. 022 = Jan 22)
		///		hh = hour (00-23)
		///		mm = minutes (00-59)
		/// </summary>
		/// <returns></returns>
		public static Regex ISBT128ExpirationDate()
		{
			return new Regex("&>[0-9]{6,10}");
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/4/2003</CreationDate>
		///<TestCases>
		///
		///<Case type="0" testid ="3446"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Regex</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="560"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		///
		/// <summary>
		/// This method returns the regular expression used to determine if Patient's SSN is valid.
		/// </summary>
		/// <returns></returns>
		public static Regex PatientSsn()
		{
			return new Regex(@"^[0-9]{4,9}([P]?)$");
		}
		
		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/4/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3447"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Regex</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="563"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NAx</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns the regular expression used to determine if Patient's Name is valid.
		/// </summary>
		/// <returns></returns>
		public static Regex PatientName()
		{
			return new Regex(@"^([A-Z]+)$");
		}
		
		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/4/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3449"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Regex</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="564"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns the regular expression used to determine if Patient's combination of
		/// Last Name Initial and last four digits of SSN is valid.
		/// </summary>
		/// <returns></returns>
		public static Regex PatientInitialAndLastFourSSN()
		{
			return new Regex(@"^[A-Za-z]{1}[0-9]{4}$");
		}
		

		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>8/29/2005</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="8132"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>Regex object</ExpectedOutput>
		///	</Case>
		///
		///
		///<Case type="1" testid ="8133"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns the regular expression used to determine if Patient's
		/// last four digits of SSN is valid.  But what about the middle 2!!  Egads!
		/// </summary>
		/// <returns></returns>
		public static Regex LastFourSSN()
		{
			return new Regex(@"^[0-9]{4}$");
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/4/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3451"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Regex</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="565"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns the regular expression used to determine if Specimen Accession Number is valid.
		/// </summary>
		/// <returns></returns>
		public static Regex SpecimenAccessionNumber()
		{
			return new Regex("BB [0-9]{4} [0-9]+");
		}
		
		///<Developers>
		///	<Developer>Greg Lohse</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/4/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="566"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Regex</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="567"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// SpecimenUid
		/// </summary>
		/// <returns></returns>
		public static Regex SpecimenUid()
		{
			return new Regex(@"^\w{10,15}$");
		}

		///<Developers>
		///	<Developer>Margaret Jablonski</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/4/2003</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="3453"> 
		///		<ExpectedInput>None</ExpectedInput>
		///		<ExpectedOutput>Valid Regex</ExpectedOutput>
		///	</Case>
		///	
		///<Case type="1" testid ="568"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// This method returns the regular expression used to determine if Return Credit Amount is valid.
		/// </summary>
		/// <returns></returns>
		public static Regex ReturnCreditAmount()
		{
			return new Regex(@"^[0-9]+(\.[0-9]{1,2})?$", RegexOptions.None);
		}


		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>5/18/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4440"> 
		///		<ExpectedInput>Valid IP address string.</ExpectedInput>
		///		<ExpectedOutput>Valid Regex (matches).</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4441"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns regular expression validating IP address in 4 octet dotted notation.
		/// Provides full validationm, checking each octet for being in range of 0-255. 
		/// </summary>
		/// <returns>Regular expression validating IP.</returns>
		public static Regex IPAddress()
		{
			return new Regex( @"\A(((25[0-5])|(2[0-4]\d)|(1\d{2})|(0?\d{1,2}))\.){3}((25[0-5])|(2[0-4]\d)|(1\d{2})|(0?\d{1,2}))\Z", RegexOptions.Singleline | RegexOptions.CultureInvariant );
		}


		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/2/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4815"> 
		///		<ExpectedInput>Valid SSN string.</ExpectedInput>
		///		<ExpectedOutput>Regex matching string.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4816"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Returns regexp validating SSN. Requires 9-digit SSN w/o dashes or spaces.
		/// </summary>
		/// <returns>Regexp validating SSN.</returns>
		public static Regex SsnNoDashes()
		{
			return new Regex( @"\A\d{9}\Z" );
		}

		///<Developers>
		///	<Developer>Stas Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/2/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4818"> 
		///		<ExpectedInput>Valid DUZ string.</ExpectedInput>
		///		<ExpectedOutput>Regexp matches.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4820"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Initalizes regexp with pattern validating VistA user DUZ. 
		/// </summary>
		/// <returns>Regexp validating VistA user DUZ.</returns>
		public static Regex VistADuz()
		{
			return new Regex( @"\A\d{1,19}\Z" );
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4824"> 
		///		<ExpectedInput>Valid division code string.</ExpectedInput>
		///		<ExpectedOutput>Regexp matches.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4825"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		/// <summary>
		/// DivisionCode
		/// </summary>
		/// <returns></returns>
		public static Regex DivisionCode()
		{
			return new Regex( @"\A[\w\s]{1,5}\Z" );
		}

		///<Developers>
		///	<Developer>Stanislav Antropov</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>7/5/2004</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="4826"> 
		///		<ExpectedInput>Valid division name string.</ExpectedInput>
		///		<ExpectedOutput>Regexp matches.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="4827"> 
		///		<ExpectedInput>NA</ExpectedInput>
		///		<ExpectedOutput>NA</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// DivisionName
		/// </summary>
		/// <returns></returns>
		public static Regex DivisionName()
		{
			return new Regex( @"\A[\w]{1}.{0,49}\Z" );
		}

		///<Developers>
		///	<Developer>saic</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>9/24/2010</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="9141"> 
		///		<ExpectedInput>Test strings that match(positive tests) and do not match(negative tests) Regular Expression.</ExpectedInput>
		///		<ExpectedOutput>Regular Expression matches or does not match, depending on test type.</ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="9142"> 
		///		<ExpectedInput>N/A</ExpectedInput>
		///		<ExpectedOutput>N/A</ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// Will match any string that contains only an integer or decimal number
		/// CR 2252
		/// </summary>
		/// <returns>Regex</returns>
		public static Regex IntegerOrDecimal()
		{
			return new Regex(@"^\d+\z|^\d+[.]\d*\z|^\d*[.]\d+\z");
		}

		///<Developers>
		///	<Developer>Hines OIFO</Developer>
		///</Developers>
		///<SiteName>Hines OIFO</SiteName>
		///<CreationDate>3/30/2011</CreationDate>
		///<TestCases>
		///	
		///<Case type="0" testid ="9196"> 
		///		<ExpectedInput>Valid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///<Case type="1" testid ="9197"> 
		///		<ExpectedInput>Invalid</ExpectedInput>
		///		<ExpectedOutput></ExpectedOutput>
		///	</Case>
		///
		///</TestCases>
		///<Update></Update>
		///<ArchivePlan></ArchivePlan>
		///<Interfaces></Interfaces>
		///
		/// <summary>
		/// 
		/// </summary>
		/// <returns>Regex</returns>
		public static Regex Integer()
		{
			return new Regex(@"^\d+\z");
		}
	}
}



